#!/usr/bin/env python3
#
# set_DAS_day.py
# Kludged by Kevin Cole <ubuntourist@hacdc.org> 2024.05.06 (kjc)
#
# Set the text value of a specific "text pthread" source named "Day
# Number" to the number of days elapsed since April 5, 2020 -- the day
# before the first Daily Antidotee of Song (DAS) episode.
#
# Kludged together from a Lua script and a LOT of trial and error. See:
#
#   https://obsproject.com/forum/threads/obs-lua-scene-item-to-source.125312/#post-467334
#   https://github.com/exeldro/obs-lua/blob/master/refresh-browsers.lua
#   https://github.com/upgradeQ/OBS-Studio-Python-Scripting-Cheatsheet-obspython-Examples-of-API/blob/master/src/obs_event_exmpl.py
#
# After editing, restart OBS Studio and reload. (Tools -> Scripts ->
# highlight this script -> reload circular arrow button.)
#

from os.path import expanduser      # Cross-platform home directory finder
import obspython as obs
import json
import colorsys
from datetime import date

# Colors work backwards, just to be perverse, apparently.
#
#  alpha-blue-green-red
#
#  letter  = 0xFF0000FF  # DEBUG - solid RED
#  outline = 0xFF00FF00  # DEBUG - solid GREEN
#  shadow  = 0xFFFF0000  # DEBUG - solid BLUE
#

# Initial global Day Number defaults:
#
letter  = 4294967295  # FFFFFFFF

# Initial global Day Number overrides:
#
letter  = 4291100671  #FFFFC4 + Alpha 255 = #FFFFFFC4
outline = 4278190080  #000000 + Alpha 255 = #FF000000
shadow  = 4285103742  #7E7E69 + Alpha 255 = #FF7E7E69
spin    = 1           # Direction of color wheel

# Initial global Gradient defaults:
#
srgb = False
rotation = 270.0
steps = 1
from_color,             from_opacity =       4291940817, 100.0,  # FFD1D1D1
midpoint_1, to_color_1, to_opacity_1 = 50.0, 4278190080, 100.0,  # FF000000
midpoint_2, to_color_2, to_opacity_2 = 50.0, 4278190080, 100.0,  #    "
midpoint_3, to_color_3, to_opacity_3 = 50.0, 4278190080, 100.0,  #    "
midpoint_4, to_color_4, to_opacity_4 = 50.0, 4278190080, 100.0,  #    "
midpoint_5, to_color_5, to_opacity_5 = 50.0, 4278190080, 100.0,  #    "
midpoint_6, to_color_6, to_opacity_6 = 50.0, 4278190080, 100.0,  #    "
midpoint_7, to_color_7, to_opacity_7 = 50.0, 4278190080, 100.0,  #    "
midpoint_8, to_color_8, to_opacity_8 = 50.0, 4278190080, 100.0,  #    "
midpoint_9, to_color_9, to_opacity_9 = 50.0, 4278190080, 100.0,  #    "

# Initial global Gradient overrides:
#
srgb = True
from_color = 4288581841  # FF9E90D1
to_color_1 = 4293039252  # FFE29494


def to_hsv(abgr):
    """Convert Alpha-Blue-Green-Red to Hue-Saturation-Value"""
    r = (abgr & 0x000000FF) >>  0
    g = (abgr & 0x0000FF00) >>  8
    b = (abgr & 0x00FF0000) >> 16
    hsv = list(colorsys.rgb_to_hsv(r, g, b))
    return hsv

def to_abgr(hsv):
    """Convert Hue-Saturation-Value to Alpha-Blue-Green-Red"""
    r, g, b = colorsys.hsv_to_rgb(*hsv)
    abgr    = 0xFF000000
    abgr   |= (int(r) <<  0)
    abgr   |= (int(g) <<  8)
    abgr   |= (int(b) << 16)
    return abgr


def dbug():
    hsv = to_hsv(shadow)
    abgr = to_abgr(hsv)
    print(f"there: {shadow:08X} ... and ... back: {abgr:08X}")


def script_description():
    """Provide a description of this script in the Tools->Scripts dialog window"""
    desc  = """<p><strong>DAS Day Number</strong></p>"""
    desc += """<p>Set the Daily Antidote of Song (DAS) day number.</p>"""
    return desc


def on_event(event):
    """Set the DAS day when the frontend has finished loading."""
    global letter, outline, shadow, spin
    global srgb, rotation, steps
    global from_color, from_opacity
    global midpoint_1, to_color_1, to_opacity_1
    global midpoint_2, to_color_2, to_opacity_2
    global midpoint_3, to_color_3, to_opacity_3
    global midpoint_4, to_color_4, to_opacity_4
    global midpoint_5, to_color_5, to_opacity_5
    global midpoint_6, to_color_6, to_opacity_6
    global midpoint_7, to_color_7, to_opacity_7
    global midpoint_8, to_color_8, to_opacity_8
    global midpoint_9, to_color_9, to_opacity_9

    if event == obs.OBS_FRONTEND_EVENT_FINISHED_LOADING:
        BDAS    = date.fromisoformat("2020-04-05")   # Day before the first DAS (Day 0)
        today   = date.today()
        elapsed = today - BDAS
        print(f"{elapsed.days}  (2024-04-29 was Day 1485.)")  # DEBUG - show in log
        source = obs.obs_get_source_by_name("Day Number")
        settings = obs.obs_source_get_settings(source)
        obs.obs_data_set_string(settings, "text", f"Day\n{elapsed.days}")
        obs.obs_data_set_int(settings, "color",         letter)
        obs.obs_data_set_int(settings, "outline_color", outline) 
        obs.obs_data_set_int(settings, "shadow_color",  shadow)
        obs.obs_source_update(source, settings)
        obs.obs_data_release(settings)
        obs.obs_source_release(source)
#       raise Exception("Triggered when the frontend has finished loading.")


def script_load(settings):
    obs.obs_frontend_add_event_callback(on_event)


def script_tick(seconds):
    """Spin the color wheel!"""
    global letter, outline, shadow, spin
    global srgb, rotation, steps
    global from_color, from_opacity
    global midpoint_1, to_color_1, to_opacity_1
    global midpoint_2, to_color_2, to_opacity_2
    global midpoint_3, to_color_3, to_opacity_3
    global midpoint_4, to_color_4, to_opacity_4
    global midpoint_5, to_color_5, to_opacity_5
    global midpoint_6, to_color_6, to_opacity_6
    global midpoint_7, to_color_7, to_opacity_7
    global midpoint_8, to_color_8, to_opacity_8
    global midpoint_9, to_color_9, to_opacity_9

    # Day Number
    #
    source = obs.obs_get_source_by_name("Day Number")
    settings = obs.obs_source_get_settings(source)
    h, s, v = to_hsv(letter)
    s += 0.001 * spin
    if s > 1.0:
        spin *= -1
        s += 0.001 * spin
    if s <= 0:
        spin *= -1
    h += 0.001
    if h > 1.0:
        h = 0.0
#           v += 0.001
#           if v > 1.0:
#               v = 0.0
    letter = to_abgr([h, s, v])
    obs.obs_data_set_int(settings, "color",         letter)
#   obs.obs_data_set_int(settings, "outline_color", outline) 
#   obs.obs_data_set_int(settings, "shadow_color",  shadow)
    obs.obs_source_update(source, settings)
    obs.obs_data_release(settings)
    obs.obs_source_release(source)

    # Gradient
    #
    source = obs.obs_get_source_by_name("Gradient")
    settings = obs.obs_source_get_settings(source)
#   ... To Be Continued ...
    from_h = h + (1.0 / 3.0)
    to_h   = h + (2.0 / 3.0)
    from_color = to_abgr([from_h, s, v])
    to_color_1 = to_abgr([to_h, s, v])
    rotation   = (rotation + 1) % 360.0
    obs.obs_data_set_int(settings,    "steps",      1)
    obs.obs_data_set_int(settings,    "from_color", from_color)
    obs.obs_data_set_int(settings,    "to_color_1", to_color_1)
    obs.obs_data_set_double(settings, "rotation",   rotation)
    obs.obs_source_update(source, settings)
    obs.obs_data_release(settings)
    obs.obs_source_release(source)
    

##############################################################################
# 
# * Using [the ChatGPT feature of Duck Duck
#   Go!](https://duckduckgo.com/?q=gently+cycle+through+hsv+%22screen+colors%22&t=brave&atb=v361-1&ia=chat)
#   I asked:
# 
# * "Can you show me an algorithm for gently cycling through HSV colors
#   on a display screen?"
# 
# > Sure! One way to cycle through HSV colors on a display screen is to
# > use the following algorithm:
# >
# > 1. Define the initial hue, saturation, and value (brightness)
# >    values.
# > 2. Increment the hue value gradually to smoothly transition through
# >    different colors.
# > 3. Convert the HSV values to RGB values for display on the screen.
# > 4. Update the display with the new RGB color.
# > 5. Repeat the process by incrementing the hue value in a loop to
# >    create a cycling effect.
# >
# > Here is a simple Python code snippet that demonstrates this
# > algorithm:
# 
# ```python
# import colorsys
# import time
# 
# # Initial HSV values
# hue = 0
# saturation = 1.0
# value = 1.0
# 
# while True:
#     # Convert HSV to RGB
#     rgb = colorsys.hsv_to_rgb(hue, saturation, value)
#     r = int(rgb[0] * 255)
#     g = int(rgb[1] * 255)
#     b = int(rgb[2] * 255)
#     
#     # Update display with RGB color
#     # (Code to update display with RGB color goes here)
#     
#     # Increment hue for next iteration
#     hue += 0.01
#     if hue > 1:
#         hue = 0
#     
#     time.sleep(0.1)  # Adjust the delay to control the speed of color cycling
# 
# ```
# 
# > You can adjust the speed of color cycling by changing the increment
# > value for the hue and the delay in the `time.sleep()` function. This
# > algorithm will smoothly cycle through different colors in the HSV
# > color space on a display screen.
#
############################################################################## 
#
# * What if the colorspace is 32 bits?
# 
# > If you are working with a 32-bit color space, you typically have 8
# > bits each for the red, green, blue, and alpha (transparency)
# > channels. In this case, you can modify the algorithm to generate
# > colors in the RGBA format. Here's an updated Python code snippet
# > that demonstrates cycling through colors in a 32-bit color space:
# 
# ```python
# import colorsys
# import time
# 
# # Initial HSV values
# hue = 0
# saturation = 1.0
# value = 1.0
# 
# while True:
#     # Convert HSV to RGB
#     rgb = colorsys.hsv_to_rgb(hue, saturation, value)
#     r = int(rgb[0] * 255)
#     g = int(rgb[1] * 255)
#     b = int(rgb[2] * 255)
#     
#     # Set alpha channel to 255 (fully opaque)
#     a = 255
#     
#     # Combine RGB and alpha values into a 32-bit color
#     color = (a << 24) | (r << 16) | (g << 8) | b
#     
#     # Update display with 32-bit color
#     # (Code to update display with 32-bit color goes here)
#     
#     # Increment hue for next iteration
#     hue += 0.01
#     if hue > 1:
#         hue = 0
#     
#     time.sleep(0.1)  # Adjust the delay to control the speed of color cycling
# 
# ```
# 
# > In this code snippet, we set the alpha channel to 255 (fully opaque)
# > and combine the RGB and alpha values into a 32-bit color. You can
# > then update the display with the 32-bit color to cycle through
# > colors in a 32-bit color space.
# >
# > Please note that the actual implementation of updating the display
# > with 32-bit colors will depend on the specific display technology
# > and programming environment you are using. You may need to adapt the
# > code to work with the display library or hardware you are using.
#
############################################################################## 
#
# * "And finally, is there a good way to always choose secondary colors
#   that will always complement or contrast the color chosen by the
#   above?"
# 
# > One way to choose secondary colors that complement or contrast the
# > primary color chosen in the algorithm above is to use color harmony
# > principles. Color harmony refers to the pleasing arrangement of
# > colors that work well together in a design. Here are a few color
# > harmony techniques you can consider:
# 
# > 1. **Complementary Colors**: Complementary colors are opposite each
# >    other on the color wheel. Choosing a secondary color that is the
# >    complement of the primary color can create a strong contrast. For
# >    example, if the primary color is red, the complementary color
# >    would be green.
# >
# > 2. **Analogous Colors**: Analogous colors are next to each other on
# >    the color wheel. Choosing secondary colors that are adjacent to
# >    the primary color can create a harmonious and cohesive look. For
# >    example, if the primary color is blue, the analogous colors could
# >    be blue-green and blue-purple.
# >
# > 3. **Triadic Colors**: Triadic colors are evenly spaced around the
# >    color wheel. Choosing secondary colors that form a triangle with
# >    the primary color can create a vibrant and balanced color
# >    scheme. For example, if the primary color is red, the triadic
# >    colors could be green and blue.
# >
# > 4. **Split-Complementary Colors**: Split-complementary colors are a
# >    variation of the complementary color scheme. Instead of choosing
# >    the exact complement, you choose the two colors adjacent to its
# >    complement. This can create a color scheme that is both
# >    contrasting and harmonious.
# >
# > 5. **Tetradic Colors**: Tetradic colors are a color scheme with two
# >    pairs of complementary colors. This can create a rich and dynamic
# >    color palette. You can choose secondary colors that form a
# >    rectangle or square on the color wheel with the primary color.
# >
# > By applying these color harmony principles, you can choose secondary
# > colors that will complement or contrast the primary color chosen in
# > your algorithm. Experimenting with different color combinations and
# > observing how they interact can help you create visually appealing
# > designs.
# 
# * I meant, algoritmicly -- i.e. numerically, not using color names.
# 
# > To choose secondary colors algorithmically based on the primary
# > color chosen in the previous code, you can use mathematical formulas
# > to generate complementary or harmonious colors. One common approach
# > is to manipulate the hue value in the HSV color space to achieve
# > different color relationships. Here's an updated Python code snippet
# > that demonstrates how you can generate secondary colors based on the
# > primary color chosen:
# 
# ```python
# import colorsys
# import time
# 
# # Initial HSV values
# hue = 0
# saturation = 1.0
# value = 1.0
# 
# while True:
#     # Convert HSV to RGB for the primary color
#     rgb_primary = colorsys.hsv_to_rgb(hue, saturation, value)
#     r_primary = int(rgb_primary[0] * 255)
#     g_primary = int(rgb_primary[1] * 255)
#     b_primary = int(rgb_primary[2] * 255)
#     
#     # Generate complementary color by shifting the hue by 0.5 (180 degrees)
#     hue_complementary = (hue + 0.5) % 1
#     rgb_complementary = colorsys.hsv_to_rgb(hue_complementary, saturation, value)
#     r_complementary = int(rgb_complementary[0] * 255)
#     g_complementary = int(rgb_complementary[1] * 255)
#     b_complementary = int(rgb_complementary[2] * 255)
#     
#     # Update display with primary and complementary colors
#     # (Code to update display with primary and complementary colors goes here)
#     
#     # Increment hue for next iteration
#     hue += 0.01
#     if hue > 1:
#         hue = 0
#     
#     time.sleep(0.1)  # Adjust the delay to control the speed of color cycling
# 
# ```
# 
# > In this code snippet, we calculate the complementary color by
# > shifting the hue value of the primary color by 0.5 (which
# > corresponds to 180 degrees on the color wheel). This simple approach
# > generates a complementary color for each primary color chosen in the
# > algorithm.
# >
# > You can further explore other color harmony principles, such as
# > analogous colors, triadic colors, split-complementary colors, or
# > tetradic colors, by manipulating the hue values accordingly. By
# > adjusting the hue values based on these principles, you can generate
# > secondary colors that complement or contrast the primary color in a
# > numerical and algorithmic way.
#
############################################################################## 
#
# * "So, if I understand correctly, tetradic colors could be found by
#   adding 0.25, 0.50 and 0.75 to the primary hue?"
# 
# > Yes, that's correct! In a tetradic color scheme, you can find the
# > secondary colors by adding 0.25, 0.50, and 0.75 to the primary hue
# > value in the HSV color space. By doing so, you can generate a set of
# > colors that form a rectangle on the color wheel, creating a vibrant
# > and dynamic color palette.
# >
# > Here's how you can modify the previous Python code snippet to
# > include tetradic colors based on the primary hue chosen:
# 
# ```python
# import colorsys
# import time
# 
# # Initial HSV values
# hue = 0
# saturation = 1.0
# value = 1.0
# 
# while True:
#     # Convert HSV to RGB for the primary color
#     rgb_primary = colorsys.hsv_to_rgb(hue, saturation, value)
#     r_primary = int(rgb_primary[0] * 255)
#     g_primary = int(rgb_primary[1] * 255)
#     b_primary = int(rgb_primary[2] * 255)
#     
#     # Generate tetradic colors by adding 0.25, 0.50, and 0.75 to the primary hue
#     hue_tetradic1 = (hue + 0.25) % 1
#     rgb_tetradic1 = colorsys.hsv_to_rgb(hue_tetradic1, saturation, value)
#     r_tetradic1 = int(rgb_tetradic1[0] * 255)
#     g_tetradic1 = int(rgb_tetradic1[1] * 255)
#     b_tetradic1 = int(rgb_tetradic1[2] * 255)
#     
#     hue_tetradic2 = (hue + 0.50) % 1
#     rgb_tetradic2 = colorsys.hsv_to_rgb(hue_tetradic2, saturation, value)
#     r_tetradic2 = int(rgb_tetradic2[0] * 255)
#     g_tetradic2 = int(rgb_tetradic2[1] * 255)
#     b_tetradic2 = int(rgb_tetradic2[2] * 255)
#     
#     hue_tetradic3 = (hue + 0.75) % 1
#     rgb_tetradic3 = colorsys.hsv_to_rgb(hue_tetradic3, saturation, value)
#     r_tetradic3 = int(rgb_tetradic3[0] * 255)
#     g_tetradic3 = int(rgb_tetradic3[1] * 255)
#     b_tetradic3 = int(rgb_tetradic3[2] * 255)
#     
#     # Update display with primary and tetradic colors
#     # (Code to update display with primary and tetradic colors goes here)
#     
#     # Increment hue for next iteration
#     hue += 0.01
#     if hue > 1:
#         hue = 0
#     
#     time.sleep(0.1)  # Adjust the delay to control the speed of color cycling
# 
# ```
# 
# > By adding 0.25, 0.50, and 0.75 to the primary hue value, you can
# > generate a set of tetradic colors that work harmoniously together in
# > a design. This approach allows you to algorithmically create a color
# > scheme that includes primary and tetradic colors for a visually
# > appealing result.
#
##############################################################################
